unit tio;

{$mode objfpc}{$H+}

// Input/Output for tlabel,tsave and tload

interface

uses
  Classes, SysUtils, baseunix;

type
  TTapeException = class(Exception);
  TTapeEndOfTapeException = class(Exception);
  TTapeReadErrorException = class(Exception);
  TTapeFormatErrorException = class(Exception);
  TNativeUInt = ptruint;
  tTapeFile = class
    private
      internalDataSize : TNativeUInt;
    public
      dataPtr : pchar;
      dataSize: TNativeUInt;
      destructor destroy; override;
      procedure addData(p:pchar; size:TNativeUInt);
  end;

  tTapeIO = class
    private
      fileList : TStringList;
      directory : string;
      currFileNo : integer;
      mode : (mode_DIR,mode_TAPE);
    public
      constructor create(dev_or_dir : string);
      destructor destroy; override;
      function readNextFile : tTapeFile;
  end;

implementation

const
  PREINITDATASIZE = 1024 * 64;

destructor tTapeFile.destroy;
begin
  if internalDataSize > 0 then
    FreeMem(dataPtr,internalDataSize);
  inherited destroy;
end;

procedure tTapeFile.addData(p:pchar; size:TNativeUInt);
var
  allocSize : TNativeUInt;
  pTemp : pchar;
begin
  if size = 0 then exit;
  allocSize := 0;
  while internalDataSize-dataSize+allocSize < size do
      inc(allocSize,PREINITDATASIZE);
  if allocSize > 0 then
    begin
      GetMem(pTemp,allocSize);
      if dataSize > 0 then
        begin
          move(dataPtr^,pTemp^,dataSize);
          freeMem(dataPtr,internalDataSize);
        end;
      dataPtr := pTemp;
      internalDataSize := allocSize;
    end;
  pTemp := dataPtr;
  inc(pTemp,dataSize);
  move(p^,pTemp^,size);
  inc(dataSize,size);
end;


//*****************************************************************************
destructor tTapeIO.destroy;
begin
  if assigned(fileList) then fileList.free;
  inherited destroy;
end;

constructor tTapeIO.create(dev_or_dir : string);
var
  s : stat;
  sr : TSearchRec;
begin
  inherited create;
  if fpstat(dev_or_dir,s) <> 0 then
    raise TTapeException.CreateFmt('unable to open %s',[dev_or_dir]);
  if fpS_ISBLK(s.st_mode) then  // block Device
    begin
      mode :=mode_TAPE;
      raise TTapeException.Create('Tape access not yet implemented');
    end;
  fileList := TStringList.create;
  if fpS_ISDIR(s.st_mode) then  // dir
    begin               // scan all files in dir and sort these
      mode := mode_DIR;
      directory := IncludeTrailingBackslash(dev_or_dir);
      if findFirst(directory+'*',faAnyFile,sr) = 0 then
        begin
          repeat
            if (sr.Attr and faDirectory) <> faDirectory then
              fileList.add(directory+sr.name);
          until findNext(sr) <> 0;
          findClose(sr);
          fileList.sort;
        end;
    end else
      raise TTapeException.Create('dev has to be a directory or a tape device');
end;


function TTapeIO.readNextFile : tTapeFile;
var
  fileName : string;
  f : THandle;
  p : pchar;
  fSize : integer;
  bytesRead : integer;
begin
  result := nil;
  if mode = mode_DIR then
    begin
      if currFileNo < fileList.count then
        begin
          fileName := fileList[currFileNo];
          inc(currFileNo);
          result := TTapeFile.create;
          f := FileOpen(fileName,fmOpenRead);
          if f = -1 then raise TTapeReadErrorException.createFMT('unable to open %s',[fileName]);
          try
            fSize := fileSeek(f,0,fsFromEnd);  // no getfilesize ?
            if fSize = -1 then raise TTapeReadErrorException.createFMT('unable to read %s',[fileName]);
            GetMem(p,fSize);
            try
              fileSeek(f,0,fsFromBeginning);
              bytesRead := fileRead(f,p^,fSize);
              if bytesRead <> fSize then TTapeReadErrorException.createFMT('unable to read %s',[fileName]);
              result.addData(p,fSize);
            finally
              FreeMem(p,fSize);
            end;
          finally
            FileClose(f);
          end;
        end else
          raise TTapeEndOfTapeException.Create('readNextFile: reached end of tape');
    end else
    begin
      raise TTapeException.Create('readNextFile for tape not yet implemented');
    end;
end;

end.

